/*
 *	T T C P . C
 *
 * Test TCP connection.  Makes a connection on port 5001
 * and transfers fabricated buffers or data copied from stdin.
 *
 * Usable on 4.2, 4.3, and 4.1a systems by defining one of
 * BSD42 BSD43 (BSD41a)
 * Machines using System V with BSD sockets should define SYSV.
 *
 * Modified for operation under 4.2BSD, 18 Dec 84
 *      T.C. Slattery, USNA
 * Minor improvements, Mike Muuss and Terry Slattery, 16-Oct-85.
 * Modified in 1989 at Silicon Graphics, Inc.
 *	catch SIGPIPE to be able to print stats when receiver has died 
 *	for tcp, don't look for sentinel during reads to allow small transfers
 *	increased default buffer size to 8K, nbuf to 2K to transfer 16MB
 *	moved default port to 5001, beyond IPPORT_USERRESERVED
 *	make sinkmode default because it is more popular, 
 *		-s now means don't sink/source 
 *	count number of read/write system calls to see effects of 
 *		blocking from full socket buffers
 *	for tcp, -D option turns off buffered writes (sets TCP_NODELAY sockopt)
 *	buffer alignment options, -A and -O
 *	print stats in a format that's a bit easier to use with grep & awk
 *	for SYSV, mimic BSD routines to use most of the existing timing code
 * Modified by Steve Miller of the University of Maryland, College Park
 *	-b sets the socket buffer size (SO_SNDBUF/SO_RCVBUF)
 * Modified Sept. 1989 at Silicon Graphics, Inc.
 *	restored -s sense at request of tcs@brl
 * Modified Oct. 1991 at Silicon Graphics, Inc.
 *	use getopt(3) for option processing, add -f and -T options.
 *	SGI IRIX 3.3 and 4.0 releases don't need #define SYSV.
 * Modified March 2005 to support Linux 2.6 network AIO only.
 *      removed all other cruft to make AIO network code readable.
 *
 * Distribution Status -
 *      Public Domain.  Distribution Unlimited.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/resource.h>

#include <libaio.h>
#include <sdp_sock.h>

int fd;			   /* fd of network socket */

int buflen = 8 * 1024;	   /* length of buffer */
int nbuf = 2 * 1024;       /* number of buffers to send in sinkmode */
int num_sec = 0;	   /* number of seconds to run */
sig_atomic_t times_up = 0; /* used when running for -N seconds */

int bufoffset = 0;	 /* align buffer to this */
int bufalign = 16*1024;	   /* modulo this */

int options = 0;	   /* socket options */
int reuse = 0;	     /* reuse listen socket */
int one = 1;	       /* for 4.3 BSD style setsockopt() */
short port = 5001;	   /* TCP port number */
char *host;		   /* ptr to name of host */
int trans;		   /* 0=receive, !0=transmit mode */
int verify = 0; 	   /* 1=verify pattern sent */
int nodelay = 0;	   /* set TCP_NODELAY socket option */
int concur = 1;	    /* concurrent AIO's */
int buff_c = 1;	    /* number of buffers */
int rcvlowat = 0;	  /* recv low water mark */
int zcopy_thrsh = -1;      /* zcopy threshold. */
int sockbufsize = 0;	   /* socket buffer size to use */
char fmt = 'm';		   /* output format: k = kilobits, K = kilobytes,
			    *  m = megabits, M = megabytes, 
			    *  g = gigabits, G = gigabytes */
int urg_notify = 0;	/* generate notification on urgent data */

char Usage[] = "\
Usage: %s -t [-options] host\n\
       %s -r [-options]     \n\
Common options:\n\
	-v	-t: generate a verifiable pattern.\n\
		-r: verify the data stream being received.\n\
	-d	set SO_DEBUG socket option\n\
	-u      urgent data notification\n\
	-l ##	length of network read/write buffers (default 8192)\n\
	-p ##	port number to send to or listen at (default 5001)\n\
	-A ##	align the start of buffers to this modulus (default 16384)\n\
	-O ##	start buffers at this offset from the modulus (default 0)\n\
	-b ##	set SO_SNDBUF and SO_RCVBUF socket buffer size\n\
	-a ##   number of concurrent outstanding AIO's (default 1)\n\
	-x ##   number of different buffers to use. (defaults to -a value)\n\
	-z ##   zcopy threshold.\n\
	-f X	format for rate: k,K = kilo{bit,byte}\n\
				 m,M = mega{bit,byte}\n\
				 g,G = giga{bit,byte}\n\
Options specific to -t:\n\
	-n ##	number of source bufs written to network (default 2048)\n\
	-N ##	number of seconds to write to network (overrides -n)\n\
	-D	set TCP_NODELAY socket option (don't buffer TCP writes)\n\
Options specific to -r:\n\
	-R      set SO_REUSEADDR socket option\n\
	-L ##   set RCVLOWAT (recv low water mark)\n\
";	

void err(char *s)
{
	fprintf(stderr,"ttcp%s: ", trans?"-t":"-r");
	perror(s);
	fprintf(stderr,"errno=%d\n",errno);
	exit(1);
}

void mes(char *s)
{
	fprintf(stdout,"ttcp%s: %s\n", trans?"-t":"-r", s);
}

void pattern(char **cp, int cu, int cnt)
{
	while (cu-- > 0)
		while (cnt-- > 0)
			cp[cu][cnt] = 0xFF;
}

char *outfmt(double b)
{
	static char obuf[50];
	switch (fmt) {
	case 'G':
		sprintf(obuf, "%.2f GB", b / 1024.0 / 1024.0 / 1024.0);
		break;
	default:
	case 'K':
		sprintf(obuf, "%.2f KB", b / 1024.0);
		break;
	case 'M':
		sprintf(obuf, "%.2f MB", b / 1024.0 / 1024.0);
		break;
	case 'g':
		sprintf(obuf, "%.2f Gbit", b * 8.0 / 1024.0 / 1024.0 / 1024.0);
		break;
	case 'k':
		sprintf(obuf, "%.2f Kbit", b * 8.0 / 1024.0);
		break;
	case 'm':
		sprintf(obuf, "%.2f Mbit", b * 8.0 / 1024.0 / 1024.0);
		break;
	}

	return obuf;
}

static void tvsub(struct timeval *tdiff,
		  struct timeval *t1,
		  struct timeval *t0)
{

	tdiff->tv_sec = t1->tv_sec - t0->tv_sec;
	tdiff->tv_usec = t1->tv_usec - t0->tv_usec;
	if (tdiff->tv_usec < 0)
		tdiff->tv_sec--, tdiff->tv_usec += 1000000;
}

void sigpipe(int sig)
{
}

void sig_alrm_recv(int value)
{
	mes("received ALRM signal");
	times_up = 1;
}

void sig_urg_recv(int value)
{
	mes("received URG signal");
}

int main(int argc, char **argv)
{
	struct sockaddr_in sinaddr;
	struct hostent *addr;

	static struct timeval in_time, out_time;
	static struct rusage  in_usage, out_usage;
	
	struct timeval tuser;
	struct timeval tsys;
	struct timeval twall;
	uint64_t us_user;
	uint64_t us_wall;
	uint64_t us_sys;
	double realt;
	int len;

	uint64_t num_bytes = 0; /* bytes on net */
	uint64_t num_calls = 0; /* number of I/O system calls */
	char   **buffs;	 /* buffer array */

	io_context_t     io_ctx = NULL; /* must be initialized ! */
	struct io_event *events;
	struct iocb    **iocbs;
#if 0
	struct timespec  ts;
#endif
	int	      result = 0;
	int	      cu = 0;
	int	      sx = 0;
	int	      ex = 0;
	int	      ev = 0;
	int	      i;
	unsigned char    c_src;
	unsigned char    c_dst;
	/*
	 * Parse parameters.
	 */
	if (argc < 2) 
		goto usage;

	while ((i = getopt(argc, argv, 
			   "udrstvRDTb:f:l:N:n:p:A:O:a:x:z:L:")) != -1) {
		switch (i) {

		case 't':
			trans = 1;
			break;
		case 'r':
			trans = 0;
			break;
		case 'd':
			options |= SO_DEBUG;
			break;
		case 'D':
			nodelay = 1;
			break;
		case 'N':
			num_sec = atoi(optarg);
			nbuf = 0;
			break;
		case 'n':
			nbuf = atoi(optarg);
			break;
		case 'l':
			buflen = atoi(optarg);
			break;
		case 's':
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 'v':
			verify = 1;
			break;
		case 'A':
			bufalign = atoi(optarg);
			break;
		case 'O':
			bufoffset = atoi(optarg);
			break;
		case 'a':
			concur = atoi(optarg);
			/* Default -x to -a value. */
			buff_c = atoi(optarg);
			break;
		case 'x':
			buff_c = atoi(optarg);
			break;
		case 'L':
			rcvlowat = atoi(optarg);
			break;
		case 'b':
			sockbufsize = atoi(optarg);
			break;
		case 'z':
			zcopy_thrsh = atoi(optarg);
			break;
		case 'f':
			fmt = *optarg;
			break;
		case 'T':
			break;
		case 'R':
			reuse = 1;
			break;
		case 'u':
			urg_notify = 1;
			break;
		default:
			goto usage;
		}
	}

	if (trans) {
		if (optind == argc)
			goto usage;
		
		host = argv[optind];
	}
	else
		nbuf = 0;
	/*
	 * Setup connections.
	 */
	if (buff_c < concur)
		err("buffer count too small (-x must be higher than -a)");
  
	events = malloc(concur * sizeof(struct io_event));
	if (!events)
		err("events malloc");

	iocbs  = malloc(concur * sizeof(struct iocb *));
	if (!events)
		err("iocbs malloc");

	buffs = (char **)malloc(buff_c * sizeof(char *));
	if (!buffs)
		err("buffs malloc");

	for (i = 0; i < buff_c; i++) {

		if (NULL == (buffs[i] = (char *)malloc(buflen+bufalign)))
			err("malloc");

		if (bufalign != 0)
			buffs[i] += ((bufalign - 
				      ((unsigned long)buffs[i] % bufalign) + 
				      bufoffset) % bufalign);
	}

	if (trans && !verify)
		(void)pattern(buffs, buff_c, buflen);

	fprintf(stdout,	"ttcp%s: ", trans ? "-t" : "-r");
	fprintf(stdout,	"buflen = %d nbuf = %d align = %d/%d port = %d",
		buflen, nbuf, bufalign, bufoffset, port);
	if (sockbufsize)
		fprintf(stdout, " sockbufsize = %d", sockbufsize);
	fprintf(stdout, "  %s\n", trans ? host : " ");

	if (io_setup(concur + 1, &io_ctx))
		err("io_setup");

	fd = socket(AF_INET_SDP, SOCK_STREAM, 0);
	if (fd < 0)
		err("socket");
	mes("socket");

	if (reuse) {
		if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, 
			       &one, sizeof(one)) < 0)
			err("setsockopt");
	}

	if (!(0 > zcopy_thrsh)) {

		if (setsockopt(fd, SOL_SDP, SDP_ZCOPY_THRSH, &zcopy_thrsh,
			       sizeof(zcopy_thrsh)) < 0)
			err("setsockopt: zcopy_thrsh");
		mes("zcopy_thrsh");
	}

	if (sockbufsize) {
		if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, &sockbufsize,
			       sizeof(sockbufsize)) < 0)
			err("setsockopt: sndbuf");
		mes("sndbuf");

		if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &sockbufsize,
			       sizeof(sockbufsize)) < 0)
			err("setsockopt: rcvbuf");
		mes("rcvbuf");
	}

	if (rcvlowat) {
		if (setsockopt(fd, SOL_SOCKET, SO_RCVLOWAT, &rcvlowat,
			       sizeof(rcvlowat)) < 0)
			err("setsockopt: rcvlowat");
		mes("rcvlowat");
	}

	if (options)  {
		if (setsockopt(fd, SOL_SOCKET, options, &one, 
			       sizeof(one)) < 0)
			err("setsockopt");
	}

	if (urg_notify) {
		int my_pid;

		signal(SIGURG, sig_urg_recv);

		my_pid = getpid();

		if (ioctl(fd, SIOCSPGRP, &my_pid))
			err("ioctl: set sigurg");

		mes("sigurg");
	}

	signal(SIGPIPE, sigpipe);
	memset(&sinaddr, 0, sizeof(sinaddr));

	if (trans) {
		if (atoi(host) > 0 )
			sinaddr.sin_addr.s_addr = inet_addr(host);
		else {
			addr = gethostbyname(host);
			if (addr == NULL)
				err("bad hostname");

			memcpy((char *)&sinaddr.sin_addr.s_addr,
			       addr->h_addr,
			       sizeof(sinaddr.sin_addr.s_addr));
		}
		
		sinaddr.sin_family = AF_INET;
		sinaddr.sin_port   = htons(port);

		if (nodelay) {
			if (setsockopt(fd, SOL_TCP, TCP_NODELAY, 
				       &one, sizeof(one)) < 0)
				err("setsockopt: nodelay");
			mes("nodelay");
		}

		if (connect(fd, (struct sockaddr *)&sinaddr, 
			    sizeof(sinaddr) ) < 0)
			err("connect");
		mes("connect");
	}
	else {

		sinaddr.sin_port =  htons(port);
		if (bind(fd, (struct sockaddr *)&sinaddr, sizeof(sinaddr)) < 0)
			err("bind");

		if (listen(fd, 0))
			err("listen");

		fd = accept(fd, NULL, 0);
		if (fd < 0)
			err("accept");

		len = sizeof(sinaddr);
		if (getpeername(fd, (struct sockaddr *)&sinaddr, &len) < 0)
			err("getpeername");

		fprintf(stdout,"ttcp-r: accept from %s\n", 
			inet_ntoa(sinaddr.sin_addr));
	}
	
	for (i = 0; i < concur; i++) {

		iocbs[i] = malloc(sizeof(struct iocb));
		if (!iocbs[i])
			err("iocb malloc");

		memset(iocbs[i], 0, sizeof(struct iocb));

		iocbs[i]->aio_fildes  = fd;
		iocbs[i]->aio_reqprio = 0;
		iocbs[i]->u.c.nbytes  = buflen;
		iocbs[i]->u.c.offset  = 0;

		if (trans)
			iocbs[i]->aio_lio_opcode = IO_CMD_PWRITE;
		else
			iocbs[i]->aio_lio_opcode = IO_CMD_PREAD;
	}

	if (num_sec > 0) {
		struct itimerval alrm_timer;
		signal(SIGALRM, sig_alrm_recv);
		
		fprintf(stdout, "ttcp%s: timer = %dsec\n", 
			trans ? "-t" : "-r", num_sec);
		alrm_timer.it_interval.tv_sec = num_sec;
		alrm_timer.it_interval.tv_usec = 0;  
		alrm_timer.it_value.tv_sec = num_sec;
		alrm_timer.it_value.tv_usec = 0;  
		
		if (setitimer(ITIMER_REAL, &alrm_timer, NULL))
			err("setitimer");
	}

	gettimeofday(&in_time, NULL);
	getrusage(RUSAGE_SELF, &in_usage);

	errno = 0;
	/*
	 * Move data
	 */
	while (nbuf == 0 || ex < nbuf) {

		for (ev = 0; 
		     cu < concur && (0 == nbuf || sx < nbuf);
		     ev++, cu++, sx++) {

			iocbs[ev]->u.c.buf = buffs[(sx%buff_c)];
			iocbs[ev]->data    = buffs[(sx%buff_c)];
			
			if (!verify || !trans)
				continue;

			for (i = 0; i < buflen; i++)
				buffs[(sx%buff_c)][i] = \
					(((uint64_t)sx * 
					  (uint64_t)buflen) + i) % 255;
		}

		result = io_submit(io_ctx, ev, iocbs);
		if (result != ev)
			err("io_submit");
#if 0
		ts.tv_sec  = 5;
		ts.tv_nsec = 0;
#endif
		result = io_getevents(io_ctx, 1, concur, events, NULL);
		if (!result && !times_up)
			err("io_getevents");

		num_calls += 2; 

		for (ev = 0; 
		     ev < result && 0 < (long)events[ev].res;
		     num_bytes += events[ev++].res, ex++, cu--) {

			if (!verify)
				continue;

			if ((unsigned long)events[ev].data != 
			    (unsigned long)buffs[(ex%buff_c)])
				err("mismatch");
			    
			if (trans)
				continue;

			for (i = 0; i < events[ev].res; i++) {
					    
				c_src = buffs[(ex%buff_c)][i] & 0xff;
				c_dst = ((num_bytes + i) % 255) & 0xff;

				if (c_src != c_dst && c_src != 0xff)
					err("verification");
			}
		}
 
		if (0 > (long)events[ev].res)
			fprintf(stderr,	"ttcp%s: Event error <%ld> <%ld>\n",
				trans ? "-t" : "-r",
				(long)events[ev].res,
				(long)events[ev].data);
		if (ev < result)
			break;
		if (times_up)
			break;
	}
	/*
	 * Process results
	 */
	if (errno)
		err("IO");

	gettimeofday(&out_time, NULL);
	getrusage(RUSAGE_SELF, &out_usage);

	tvsub(&twall, &out_time, &in_time);
	tvsub(&tuser, &out_usage.ru_utime, &in_usage.ru_utime);
	tvsub(&tsys,  &out_usage.ru_stime, &in_usage.ru_stime);

	us_wall = (((uint64_t)twall.tv_sec * 1000000) + twall.tv_usec);
	us_user = (((uint64_t)tuser.tv_sec * 1000000) + tuser.tv_usec);
	us_sys  = (((uint64_t)tsys.tv_sec * 1000000) + tsys.tv_usec);

	realt = ((double)us_wall)/1000000;
	realt = (realt > 0.0) ? realt : 0.001;

	fprintf(stdout,
		"ttcp%s: %lld bytes in %.2f real seconds = %s/sec +++\n",
		trans ? "-t" : "-r",
		num_bytes, realt, outfmt(((double)num_bytes)/realt));
	fprintf(stdout,
		"ttcp%s: %lld I/O calls, usec/call = %.2f, calls/sec = %.2f\n",
		trans ? "-t" : "-r",
		num_calls,
		1000000.0 * realt/((double)num_calls),
		((double)num_calls)/realt);
	fprintf(stdout,
		"ttcp%s: user: %llu sys: %llu total: %llu real: %llu "
		"(microseconds)\n",
		trans ? "-t" : "-r",
		(unsigned long long)us_user,
		(unsigned long long)us_sys,
		(unsigned long long)(us_user + us_sys),
		(unsigned long long)us_wall);


	return 0;
usage:
	fprintf(stderr, Usage, argv[0], argv[0]);
	return 1;
}
